"use client";

import * as React from "react";

import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import { Button } from "@/components/ui/button";
import { Text } from "@/components/typeography/text"
import { cn } from "@/lib/utils";
import { Check, ChevronsUpDown, X } from "lucide-react";
import { ScrollArea } from "./scroll-area";
import { focusRing } from "@/lib/styles";

export interface SelectMultiPropsOption {
  label: string;
  value: string;
  icon?: React.ComponentType<{ className?: string }>;
}

interface SelectMultiProps {
  className?: string;
  defaultOptions?: string[],
  selectedOptions?: string[],
  onSelectionChanged?: (value: string[]) => void;
  onSelected?: (value: string) => void;
  options: SelectMultiPropsOption[];
}

export function SelectMulti({
  options,
  className,
  selectedOptions,
  defaultOptions = [],
  onSelected,
  onSelectionChanged,
}: SelectMultiProps) {
  const [selectedValues, setSelectedValues] = React.useState<string[]>(selectedOptions ?? defaultOptions);

  React.useEffect(() => {
    if (!selectedOptions) {
      setSelectedValues(defaultOptions)
      return;
    }

    setSelectedValues(selectedOptions)
  }, [defaultOptions, selectedOptions])

  return (
    <Popover>
      <div className="flex flex-row">
        <PopoverTrigger asChild>
          <Button
            variant="outline"
            size="sm"
            className={cn(
              "flex-1 w-full flex flex-row border-input py-2 text-foreground justify-center",
              "rounded-e-none border border-border shadow-sm ",
              selectedValues?.length > 0 ? "justify-normal" : "justify-center",
              className)}
          >
            <div className="flex-1 flex flex-row">
              {selectedValues.length === 1 ? (
                <Text
                  variant={"p"}
                  affects={"xsmall"}
                  override={'removeMarginsAndLeading'}
                  className="px-1 font-normal line-clamp-1 text-start break-all"
                >
                  {options.find((o) => o.value === selectedValues[0])?.label}
                </Text>

              ) : selectedValues.length > 2 ? (
                <>
                  <Text
                    variant={"p"}
                    affects={"xsmall"}
                    override={'removeMarginsAndLeading'}
                    className="line-clamp-1 max-w-64 text-wrap text-start break-all"
                  >
                    {options.find((o) => o.value === selectedValues[0])?.label}
                  </Text>
                  <Text
                    variant={"p"}
                    affects={"xsmall"}
                    override={'removeMarginsAndLeading'}
                    className="px-1 font-normal line-clamp-1 text-start break-all"
                  >
                    and {selectedValues.length} more
                  </Text>
                </>
              ) : (
                selectedValues
                  .map((value, i) => {
                    const option = options.find((o) => o.value === value)!
                    return (
                      <>
                        <Text
                          key={option.value}
                          variant={"p"}
                          affects={"xsmall"}
                          override={'removeMarginsAndLeading'}
                          className="line-clamp-1 max-w-64 text-wrap text-start break-all"
                        >
                          {option.label}
                        </Text>

                        <Text
                          variant={"p"}
                          affects={"xsmall"}
                          override={'removeMarginsAndLeading'}
                          className="truncate line-clamp-1 text-wrap text-start pr-2"
                        >
                          {i !== selectedValues.length - 1 && (
                            <>,{" "}</>
                          )}
                        </Text>
                      </>
                    )
                  })
              )}
            </div>

            <ChevronsUpDown className="size-3 text-muted-foreground" />
          </Button>
        </PopoverTrigger>

        <Button
          type="reset"
          onClick={() => {
            const newSelection: string[] = defaultOptions;
            setSelectedValues(newSelection);
            onSelectionChanged?.(newSelection);
          }}
          variant={"outline"}
          size="sm"
          className={cn(
            focusRing,
            "rounded-none shadow-sm rounded-s-none rounded-e-lg border border-border border-l-transparent",
            "group text-muted-foreground focus-visible:!text-destructive focus-visible:!border-destructive ring-destructive/20",
          )}
        >
          <X className="size-3" />
        </Button>
      </div>

      <PopoverContent
        align="start"
        className="p-1"
        onWheel={(e) => {
          // Popover issue: https://github.com/radix-ui/primitives/issues/1159
          e.stopPropagation();
        }}
      >
        <ScrollArea className="pr-2 h-64 gap-0">
          {options.map((option) => {
            const isSelected = selectedValues.includes(option.value);
            return (
              <Button
                variant={"ghost"}
                size={"sm"}
                className="grid grid-cols-[auto,_auto,_1fr] p-2 rounded-sm justify-start items-center hover:bg-muted w-full"
                key={option.value}
                onClick={() => {
                  const newSelection: string[] = isSelected
                    ? selectedValues.filter((v) => v !== option.value)
                    : [...selectedValues, option.value];

                  setSelectedValues(newSelection);
                  onSelectionChanged?.(newSelection);
                  onSelected?.(option.value);
                }}
              >
                <Check className={cn(
                  "mr-2 size-3 text-primary",
                  isSelected ? "" : "text-transparent"
                )} />

                {option.icon && (
                  <option.icon className="mr-2 h-4 w-4 text-muted-foreground" />
                )}

                <span className={cn(
                  "text-xs text-start truncate break-all"
                )}
                >{option.label}</span>
              </Button>
            );
          })}
        </ScrollArea>

        {selectedValues.length > 0 && (
          <>

            <div className="pt-2 grid">
              <Button
                variant={"secondary"}
                onClick={() => {
                  const newSelection: string[] = defaultOptions;
                  setSelectedValues(newSelection);
                  onSelectionChanged?.(newSelection);
                }}
                className="justify-center text-center"
              >
                Clear filters
              </Button>
            </div>
          </>
        )}
      </PopoverContent>
    </Popover >
  );
}

